/*
  U.S. Road sign library
  
  Dependencies: Requires the Roadgeek font set for text, available at: http://miketheactuary.wordpress.com/roadgeek-fonts/
                The fonts need to be converted to TTF format before use. If you have problems with incorrect characters
                showing up on your signs, try adding "global_settings{charset utf8}" to your scene file.

*/



     #declare bgsgreen=pigment{rgb <0,110,85>/255};     //green color for sign backgrounds


//         
//  twodigitshield(image)
//    Produces a two-digit road shield, given an image file in PNG format. PNG image should have square aspect ratio.
//    Shield size is 10x10 Povray units; needs to be scaled to be used as a road sign.       
         #macro twodigitshield(image)
              box{<-.325,-5,-5>,<0,5,5> pigment{image_map{png image filter all .1} rotate<0,-90,0> translate <0,-.5,-.5>  matrix<1, 0, 0,
                                               0, 1, 0,
                                               0, 0, -1,
                                               0, 0, 0 > scale <1,10,10>} rotate<0,-90,0>}
         #end    
 
 
//
//  threedigitshield(image)
//  Similar to twodigitshield, except uses a 1:1.25 aspect ratio. Shield size is 12.5x10 Povray units.      
         #macro threedigitshield(image)
              box{<-.325,-5,-6.25>,<0,5,6.25> pigment{image_map{png image filter all .1} rotate<0,-90,0> translate <0,-.5,-.5>  matrix<1, 0, 0,
                                               0, 1, 0,
                                               0, 0, -1,
                                               0, 0, 0 > scale <1,10,12.5>} rotate<0,-90,0>}
         #end


//  itemwidth(item)
//    Returns the width of the object 'item'.          
         #macro itemwidth(item)
             (max_extent(item)-min_extent(item)).x
         #end 
          

//  bgs(shieldimgs,numdigits,texts,includeexit,exitnum,arroworexit,arrowcharorexitnum,exitdistunits,clearview)
//   Produces a "big green sign" (a freeway guide sign) with specified text, shields, and arrows.
//      Example:  bgs(array[2]{"US77.png","SOUTH"},array[2]{2,0},array[1]{"Norman"},1,"113",0,"2","MILES",0);
//   Arguments:
//      shieldimgs is an array of shield image filenames and texts to display on the same level as the shields.
//      numdigits is an array (of the same size as shieldimgs) that specifies the size of the shields and whether to instead show text.
//           2 gives a square shield, 3 gives a wide shield, and 0 displays text
//      texts is an array which contains the destination(s) to list on the sign. Each element in the array is displayed on a new line
//      includeexit is a number (1 or 0) which specifies whether to include an exit number on the sign (1 to include, 0 to not include)
//      exitnum is a string which contains the exit number
//      arroworexit is a number (1 or 0) which decides whether the sign shows an arrow pointing to the exit, or a distance to the exit (1 for arrow, 0 for dist)
//      arrowcharorexitnum is a string that contains either the Roadgeek font arrow character, or the distance number, depending on arroworexit
//      exitdistunits is the unit text for the distance (typically "MILES")
//      clearview is a number (1 or 0) which dictates whether the new Clearview font, or the traditional FHWA Gothic font is used on the sign.
//
//   The above example produces a sign for Exit 113, which displays a US 77 shield and "SOUTH" on the first line, "Norman" on the second,
//      and "EXITS 2 MILES" at the bottom of the sign.                
           #macro bgs(shieldimgs,numdigits,texts,includeexit,exitnum,arroworexit,arrowcharorexitnum,exitdistunits,clearview)
             union{ 
               #local lsh=dimension_size(shieldimgs,1);
               #local lnd=dimension_size(numdigits,1);
               #local ltx=dimension_size(texts,1);    
                                     
               
               #local totalshieldwidth=0;
               #local shieldpositions=array[min(lsh,lnd)];

               #if(clearview=1)
                  #local typeface="Roadgeek 2005 Series 5W.ttf";
               #else
                  #local typeface="Roadgeek 2005 Series EM.ttf";
               #end
               
               #local maxtexwid=0;
               
               #local ntx=0;
               #while (ntx<ltx)
                  #local thetex=text{ttf typeface texts[ntx] 0.0625 0 pigment{White} scale 4 translate <0,-1-ntx*4,-.3125>}
                  object{thetex translate (-itemwidth(thetex)/2*x)}
                  #if (itemwidth(thetex)>maxtexwid) #local maxtexwid=itemwidth(thetex); #end
                  //#error str(itemwidth(thetex),0,3)
                  #local ntx=ntx+1;
               #end
               
               union{
                #local shn=0;
                #while(shn<min(lsh,lnd))
                   #if(numdigits[shn]=1 | numdigits[shn]=2) 
                      //#error str(shn,0,0)  
                      #local theobj=object{twodigitshield(shieldimgs[shn]) scale .5 translate<4+totalshieldwidth/2,6.0,-.15>};
                      #local shieldpositions[shn]=totalshieldwidth;    
                      //#error str(itemwidth(theobj),0,3)
                      #local totalshieldwidth=totalshieldwidth+itemwidth(theobj)*2+2;
                      object{theobj}
                   #else 
                    #if(numdigits[shn] >2 )   
                      #local theobj=object{threedigitshield(shieldimgs[shn]) scale .5 translate<4+totalshieldwidth/2,6.0,-.15>}
                      #local shieldpositions[shn]=totalshieldwidth;
                      #local totalshieldwidth=totalshieldwidth+itemwidth(theobj)*2+2;
                      object{theobj}
                    #end
                   #end
                   #if(numdigits[shn]=0)
                      #local theobj=text{ttf "Roadgeek 2005 Series EM.ttf" shieldimgs[shn] 0.0625 0 pigment{White} scale 8 scale .5 translate<4+totalshieldwidth/2,4.5,-.15>}
                      #local shieldpositions[shn]=totalshieldwidth;
                      #local totalshieldwidth=totalshieldwidth+itemwidth(theobj)*2+2;
                      object{theobj translate -itemwidth(theobj)/8*x}
                   #end
                 
                   #local shn=shn+1;
                #end  
                translate -totalshieldwidth/4*x
               }
               
               #local maxtexwid=max(maxtexwid,totalshieldwidth/2);
               
               #if (arroworexit=1)
                  text{ttf "Roadgeek 2005 Arrows 1.ttf" arrowcharorexitnum 0.0625 0 pigment{White} scale 8 translate <maxtexwid/2+2,3-ntx*2,-.3125>} 
                  #local rightsideadjust=5;
                  #local leftsideadjust=0;
               #end     
               #if (arroworexit=-1)
                  #local thearrow=text{ttf "Roadgeek 2005 Arrows 1.ttf" arrowcharorexitnum 0.0625 0 pigment{White} scale 8 translate <-maxtexwid/2-2,3-ntx*2,-.3125>}
                  object{thearrow translate -itemwidth(thearrow)*x} 
                  #local rightsideadjust=0;
                  #local leftsideadjust=5;
               #end 
               #if (arroworexit=0)
                  #local thetex1=text{ttf "Roadgeek 2005 Series EM.ttf" concat("EXITS ") 0.0625 0 pigment{White} scale 3 translate <0,-1-ntx*4,-.3125>}
                  #local thetex2=text{ttf "Roadgeek 2005 Series EM.ttf" concat(arrowcharorexitnum) 0.0625 0 pigment{White} scale 4 translate <itemwidth(thetex1)+1,-1-ntx*4,-.3125>}
                  #local thetex3=text{ttf "Roadgeek 2005 Series EM.ttf" concat(" ",exitdistunits) 0.0625 0 pigment{White} scale 3 translate <itemwidth(thetex1)+itemwidth(thetex2)+1,-1-ntx*4,-.3125>}
                  #local thetex=union{object{thetex1} object{thetex2} object{thetex3}};
                  object{thetex translate -itemwidth(thetex)/2*x}                                         //,arrowcharorexitnum," MILES"
                  #if (itemwidth(thetex)>maxtexwid) #local maxtexwid=itemwidth(thetex); #end
                  #local rightsideadjust=0;
                  #local leftsideadjust=0;
                  #local ltx=ltx+1;
               #end
               
               //#local backing=box{<-maxtexwid/2-2,2-4*ltx,0>,<maxtexwid/2+2+rightsideadjust,10,2> pigment{rgb <0,110,85>/255}} 
               #local ring=   box{<-maxtexwid/2-2-leftsideadjust,2-4*ltx,0>-<.5,.5,.1>,<maxtexwid/2+2+rightsideadjust,10,.2>+<.5,.5,.05>}// pigment{rgb <0,110,85>/255}} 
               #local grring= box{<-maxtexwid/2-2-leftsideadjust,2-4*ltx,0>-<1,1,0>,<maxtexwid/2+2+rightsideadjust,10,.2>+<1,1,0>}//  pigment{White}} 
               
               union{
                object{grring pigment{White}}
                object{ring pigment{bgsgreen}}
                object{grring translate .25*z texture{Aluminum finish{reflection 0.025}}} //backing
               }
               
               union{  //exit tab
                #if(includeexit=1)
                  #local thetex1=text{ttf "Roadgeek 2005 Series EM.ttf" "EXIT" 0.0625 0 pigment{White} scale 3 translate <0,12,-.3125>}    
                  #local thetex2=text{ttf "Roadgeek 2005 Series EM.ttf" concat(" ",exitnum) 0.0625 0 pigment{White} scale 4 translate <itemwidth(thetex1),12,-.3125>}
                  #local thetex=union{object{thetex1} object{thetex2}};
                  object{thetex translate (maxtexwid/2+6.5-itemwidth(thetex))*x+.25*y}
                  
                  box{<maxtexwid/2+8-itemwidth(thetex)-3,11,0>,<maxtexwid/2+8,16,.2> pigment{White}}
                  box{<maxtexwid/2+8-itemwidth(thetex)-2.5,11.5,-.05>,<maxtexwid/2+7.5,15.5,.25> pigment{rgb <0,110,85>/255}} 
                  box{<maxtexwid/2+8-itemwidth(thetex)-3,11,0>,<maxtexwid/2+8,16,.2> translate .25*z texture{Aluminum finish{reflection .025}}} //backing
                #end 
                translate (rightsideadjust-5)*x
               }
             } 
             
           #end 
    
    
//  signblade(streetname,streettype,typescale,typeoffset,usesecondlabel,secondlabel,secondlabeloffset,includelogo,logoimg,logoscale,colr,clearview)
//
//  Produces a street name sign.
//  
//  Example: signblade("LINDSEY ST."," 500",.5,0,1," WEST",3,1,"Normanblue.png",1,pigment{rgb <1,0,0>},0)
//  
//  Arguments:
//     streetname is a string that contains the name of the street.
//     streettype is a string that typically contains the street type (e.g. "Road", "Street", "Avenue", etc.)
//     typescale is a scale factor for streettype
//     typeoffset is an offset distance for streettype
//     usesecondlabel is a number (1 or 0) which decides whether a second type label will be displayed (for example for a block number or direction)
//     secondlabel is the text to display in the second label.
//     secondlabeloffset is an offset distance for secondlabel.
//     includelogo is a number (1 or 0) which decides whether a logo image is included on the sign.
//     logoimg is a filename for a PNG image for a logo
//     logoscale is a scale factor for the logo image
//     colr is the background color for the sign (typically bgsgreen)
//     clearview  is a number (1 or 0) which dictates whether the new Clearview font, or the traditional FHWA Gothic font is used on the sign.
//
//  In the above example, the code produces a street sign for W. Lindsey St., using the style used by the city of Norman, Oklahoma.
       
         #macro signblade(streetname,streettype,typescale,typeoffset,usesecondlabel,secondlabel,secondlabeloffset,includelogo,logoimg,logoscale,colr,clearview)
            #if(clearview=1)
                  #local typeface="Roadgeek 2005 Series 3W.ttf";
               #else
                  #local typeface="Roadgeek 2005 Series C.ttf";
               #end
            
           #local blade= union{  
             #if(includelogo=1)
                #local thetex3=box{<-.325,0,0>,<0,8,5> pigment{image_map{png logoimg once filter all .1} rotate<0,-90,0> scale logoscale translate <0,0,-1> scale <1,8,-5>} rotate<0,-90,0> translate 3*x}
                #local tx3wid=itemwidth(thetex3);
             #else
                #local tx3wid=0;
             #end 
             
             #local thetex1=text{ttf typeface streetname 0.0625 0 pigment{White} scale 6 translate <tx3wid,1.5,-.3125>}    
             #local thetex2=text{ttf typeface streettype 0.0625 0 pigment{White} scale 6*typescale translate <tx3wid+itemwidth(thetex1)+1,1.5+typeoffset,-.3125>}    
             #if (usesecondlabel)
               #local thetex4=text{ttf typeface secondlabel 0.0625 0 pigment{White} scale 6*typescale translate <tx3wid+itemwidth(thetex1)+1,1.5+secondlabeloffset,-.3125>}    
             #end
            
             #local thetex=union{object{thetex1} object{thetex2} #if(includelogo=1) object{thetex3} #end #if(usesecondlabel) object{thetex4} #end};
             
             object{thetex}
             
             #local texwid=itemwidth(thetex);
             
             box{<-2,0,0>,<texwid+2,8,.125> pigment{colr}}
             
             translate -.125*z
            };
            
            union{
             object{blade translate (-itemwidth(blade)/2+2)*x}
             object{blade translate (-itemwidth(blade)/2+2)*x rotate 180*y} 
            }
         
         #end


//  normanredsignblade(streetname,number,nsew)
//
//  A shortcut definition for producing Norman, Oklahoma, street signs.
//
//  Arguments:
//     streetname, String, the name of the street
//     number, String, the block number
//     nsew, String, the direction.
//
//  For example, to produce the W. Lindsey St. definition from above: normanredsignblade("LINDSEY ST. "," 500"," WEST")
         
         #macro normanredsignblade(streetname,number,nsew)
            object{signblade(streetname,number,.5,0,1,nsew,3,1,"Normanblue.png",1,pigment{rgb <1,0,0>},0)}
         #end

//  normangreensignblade(streetname,number,nsew)
//
//  A shortcut definition for producing new-style green Norman, Oklahoma, street signs.
//
//  Arguments:
//     streetname, String, the name of the street
//     number, String, the block number
//     nsew, String, the direction.
//
//  For example, to produce the W. Lindsey St. definition from above: normanredsignblade("LINDSEY ST. "," 500"," WEST")
         
         #macro normangreensignblade(streetname,number,nsew)
            object{signblade(streetname,number,.5,0,1,nsew,3,1,"Norman.png",1,bgsgreen,0)}
         #end 


//  speedlimitsign(limit)
//
//  Produces a speed limit sign, with the limit specified in the string 'limit'.         
         #macro speedlimitsign(limit)
            union{
              #local thetex1=text{ttf "Roadgeek 2005 Series EM.ttf" "SPEED" 0.0625 0 pigment{Black} scale 6 translate <0,22,-.3125>}
              #local thetex2=text{ttf "Roadgeek 2005 Series EM.ttf" "LIMIT" 0.0625 0 pigment{Black} scale 6 translate <0,16,-.3125>}
              #local thetex3=text{ttf "Roadgeek 2005 Series EM.ttf" limit 0.0625 0 pigment{Black} scale <14,14,1> translate <0,3.5,-.3125>}
              #local thetex1=object{thetex1 translate (12-itemwidth(thetex1)/2)*x}
              #local thetex2=object{thetex2 translate (12-itemwidth(thetex2)/2)*x}
              #local thetex3=object{thetex3 translate (12-itemwidth(thetex3)/2)*x}
              #local thetex=union{object{thetex1} object{thetex2} object{thetex3}};
              
              object{thetex}
              
              box{<0,0,0>,<24,30,.125> pigment{White}} 
              box{<0,0,0>,<24,30,.125> translate .125*z texture{Aluminum finish{reflection .025}}}
              difference{
                box{<0,0,0>+<.5,.5,-.01>,<24,30,.125>-<.5,.5,-.01> pigment{Black}}
                box{<0,0,0>+<1,1,-.02>,<24,30,.125>-<1,1,-.02> pigment{White}}
              } 
              translate -12*x
            }
         #end  

//  oregonspeedlimitsign(limit)
//
//  Produces an Oregon-style speed limit sign, with the limit specified in the string 'limit'.         
         #macro oregonspeedlimitsign(limit)
            union{
              #local thetex2=text{ttf "Roadgeek 2005 Series C.ttf" "SPEED" 0.0625 0 pigment{Black} scale 8 translate <0,21,-.3125>}
              //#local thetex2=text{ttf "Roadgeek 2005 Series EM.ttf" doublestring("SPEED") 0.0625 0 pigment{Black} scale 6 translate <0,16,-.3125>}
              #local thetex3=text{ttf "Roadgeek 2005 Series C.ttf" limit 0.0625 0 pigment{Black} scale 22 translate <0,3.5,-.3125>}
              //#local thetex1=object{thetex1 translate (12-itemwidth(thetex1)/2)*x}
              #local thetex2=object{thetex2 translate (12-itemwidth(thetex2)/2)*x}
              #local thetex3=object{thetex3 translate (12-itemwidth(thetex3)/2)*x}
              #local thetex=union{object{thetex2} object{thetex3}};
              
              object{thetex}
              
              box{<0,0,0>,<24,30,.125> pigment{White}}
              difference{
                box{<0,0,0>+<.5,.5,-.01>,<24,30,.125>-<.5,.5,-.01> pigment{Black}}
                box{<0,0,0>+<1,1,-.02>,<24,30,.125>-<1,1,-.02> pigment{White}}
              }
            }
         #end 